# -*- coding: utf-8 -*-
from PyQt5.QtCore import *
from PyQt5.QtGui import *
import glob
import math
import os
import shutil

from PyQt5 import QtCore
from PyQt5 import QtWidgets

from qgis.PyQt.QtCore import *
from qgis.core import *
import qgis.utils


__all__ = ['layerExists', 'getFieldIndexFromLayer', 'getUniqueForestryPrefixes', 'createPrjFile', 'getPpozFeatures', 'createPpozMemLayer',
           'copyQmlStyle', 'createSimplePointMemLayer', 'createSimpleLineMemLayer', 'createSimplePolyMemLayer', 'createPpozMemFireLayer',
           'updateFeatureInLayer', 'isFloat', 'locatePointInPolyLayer', 'degMinSec', 'transformGeometry', 'pointAlongLine']

class GeneralUtils:

    def __init__(self):
       pass

    @staticmethod
    def layerExists(path, name):
        return QtCore.QFileInfo(os.path.join(str(path), name + ".shp")).exists() & QtCore.QFileInfo(os.path.join(str(path), name + ".dbf")).exists() & QtCore.QFileInfo(os.path.join(str(path), name + ".shx")).exists()

    @staticmethod
    def getFieldIndexFromLayer(fieldName, layer):
        fldIndex = layer.dataProvider().fieldNameIndex(fieldName)
        return fldIndex

    @staticmethod
    def getUniqueForestryPrefixes(features):
        nadl = set()
        for feature in features:
            nadl.add(feature["adr_les"].split('-')[0] + feature["adr_les"].split('-')[1])

        return nadl

    @staticmethod
    def createPrjFile(dataCatalog):
        for shp in glob.glob(os.path.join(str(dataCatalog), '*.shp')):
            prjFile = shp.replace('shp','prj')
            if not os.path.exists(prjFile):
                outputFile = open(prjFile,"w")
                textPrj = """PROJCS["ETRS89_Poland_CS92",GEOGCS["GCS_ETRS89",DATUM["D_ETRS_1989",SPHEROID["GRS_1980",6378137,298.257222101]],PRIMEM["Greenwich",0],UNIT["Degree",0.017453292519943295]],PROJECTION["Transverse_Mercator"],PARAMETER["latitude_of_origin",0],PARAMETER["central_meridian",19],PARAMETER["scale_factor",0.9993],PARAMETER["false_easting",500000],PARAMETER["false_northing",-5300000],UNIT["Meter",1]]"""
                outputFile.write(textPrj)
                outputFile.close()

    @staticmethod
    def copyQmlStyle(dataCatalog, qmlCatalog):
        listaShp = glob.glob(os.path.join(str(dataCatalog), '*.shp'))
        listaQml = glob.glob(os.path.join(str(qmlCatalog), '*.qml'))

        for Shp in listaShp:
            shpPath,shpTail = os.path.split(Shp)
            shpName, shpExt = os.path.splitext(shpTail)
            for Qml in listaQml:
                qmlPath, qmlTail = os.path.split(Qml)
                qmlName, qmlExt = os.path.splitext(qmlTail)
                if str(shpName) == qmlName:
                    #if not os.path.exists(os.path.join(unicode(dataCatalog),qmlName + qmlExt)):
                    shutil.copy(Qml, dataCatalog)
    
    @staticmethod
    def getPpozFeatures(ppozLayer, forestryPrefixes):
        ppozFeatures = set()
        for val in list(forestryPrefixes):
            # nadlesnictwa z prefixem 0x-xx
            if val.startswith("0"):
                val = val[1:]
            # wydobywamy obiekty z pasujacym nadlesnictwem
            expr = qgis.core.QgsExpression(
            "nadl like '%" + val + "' or nadl like '" + val + ",%' or nadl like '%," + val + ",%' or nadl like '%," + val + "'")
            for feature in ppozLayer.getFeatures(qgis.core.QgsFeatureRequest(expr)):
                ppozFeatures.add(feature)
        return ppozFeatures

    @staticmethod
    def getParkiFeatures(pnLayer, forestryPrefixes):
        ppozFeatures = set()
        for val in list(forestryPrefixes):
            # nadlesnictwa z prefixem 0x-xx
            if val.startswith("0"):
                val = val[1:]
            # wydobywamy obiekty z pasujacym nadlesnictwem
            expr = qgis.core.QgsExpression(
                "nadl like '%" + val + "' or nadl like '" + val + ",%' or nadl like '%," + val + ",%' or nadl like '%," + val + "'")
            for feature in pnLayer.getFeatures(qgis.core.QgsFeatureRequest(expr)):
                ppozFeatures.add(feature)
        return ppozFeatures

    @staticmethod
    def getParkiOddzWydzFeatures(parkiMemLayer, pnOddzLayer):
        ppozFeatures = set()
        idx = parkiMemLayer.fields().indexFromName('PARK_NR')
        for feat in parkiMemLayer.getFeatures():
            nr = feat.attributes()[idx]
            print(nr)
            expr = qgis.core.QgsExpression(
            "PARK_NR = " + str(nr))
            for feature in pnOddzLayer.getFeatures(qgis.core.QgsFeatureRequest(expr)):
                ppozFeatures.add(feature)
        return ppozFeatures
    @staticmethod
    def createPpozMemLayer(ppozLayer, ppozFeatures, layerName, svgCatalog = "/python/plugins/mlppoz/svg"):
        newLayer = QgsVectorLayer("Point?crs=EPSG:2180", layerName, "memory")
        #newLayer.dataProvider().setEncoding(u'System')
        # etykiety
        layer_settings  = QgsPalLayerSettings()
        text_format = QgsTextFormat()

        text_format.setFont(QFont("Arial", 10))
        text_format.setSize(10)

        buffer_settings = QgsTextBufferSettings()
        buffer_settings.setEnabled(True)
        buffer_settings.setSize(0.10)
        buffer_settings.setColor(QColor("black"))
        layer_settings.labeling="pal"
        text_format.setBuffer(buffer_settings)
        layer_settings.setFormat(text_format)
        layer_settings.fieldName = "nzw_ob"
        layer_settings.placement = qgis.core.QgsPalLayerSettings.AroundPoint
        layer_settings.dist=3
        layer_settings.enabled = True

        layer_settings = QgsVectorLayerSimpleLabeling(layer_settings)
        newLayer.setLabelsEnabled(True)
        newLayer.setLabeling(layer_settings)
        newLayer.triggerRepaint()

        # symbole
        svgStyle = {}
        svgStyle['fill'] = '#ff0000'
        svgStyle['name'] = os.path.join(QtCore.QFileInfo(qgis.core.QgsApplication.qgisUserDatabaseFilePath()).path() + str(svgCatalog), "tower.svg")
        svgStyle['outline'] = '#ff0000'
        svgStyle['outline-width'] = '6.8'
        svgStyle['size'] = '8'

        symLyr1 = QgsSvgMarkerSymbolLayer.create(svgStyle)
        newLayer.renderer().symbols(QgsRenderContext())[0].changeSymbolLayer(0, symLyr1)
        # pola zrodlowej warstwy
        inFields = ppozLayer.dataProvider().fields()
        # pola docelowej warstwy
        newLayer.dataProvider().addAttributes(inFields.toList())
        # skopiowanie features z ppozLayer
        newLayer.startEditing()
        newLayer.dataProvider().addFeatures(list(ppozFeatures))
        newLayer.commitChanges()
    
        return newLayer

    @staticmethod
    def createPnMemLayer(pnLayer, ppozFeatures, layerName,type=None):
        if type is None:
            newLayer = qgis.core.QgsVectorLayer("Polygon?crs=EPSG:2180", layerName, "memory")
        else:
            newLayer = qgis.core.QgsVectorLayer(type+"?crs=EPSG:2180", layerName, "memory")

        # pola zrodlowej warstwy
        inFields = pnLayer.dataProvider().fields()
        # pola docelowej warstwy
        newLayer.dataProvider().addAttributes(inFields.toList())
        # skopiowanie features z ppozLayer
        newLayer.startEditing()
        newLayer.dataProvider().addFeatures(list(ppozFeatures))
        newLayer.commitChanges()
        return newLayer

    @staticmethod
    def createPpozMemFireLayer(layerName, color, svgCatalog = "/python/plugins/mlppoz/svg/"):
        newLayer = qgis.core.QgsVectorLayer("Point?crs=EPSG:2180", layerName, "memory")
        #newLayer.dataProvider().setEncoding(u'System')
        # symbole
        svgStyle = {}
        svgStyle['fill'] = '#ffffff'
        svgStyle['name'] = os.path.join(QtCore.QFileInfo(qgis.core.QgsApplication.qgisUserDatabaseFilePath()).path() + str(svgCatalog), "fire.svg")
        svgStyle['outline'] = '#ffffff'
        svgStyle['outline-width'] = '6.8'
        svgStyle['size'] = '6'
        symLyr1 = qgis.core.QgsSvgMarkerSymbolLayer.create(svgStyle)
        newLayer.renderer().symbols(QgsRenderContext())[0].changeSymbolLayer(0, symLyr1)
        '''symbol = QgsMarkerSymbolV2.createSimple({'name': 'diamond', 'color': color, 'size': '5'})
        newLayer.rendererV2().setSymbol(symbol)'''
        newLayer.dataProvider().addAttributes([qgis.core.QgsField("ID", QVariant.Int)])

        return newLayer

    @staticmethod
    def createSimplePointMemLayer(layerName, color, svgCatalog = "/python/plugins/mlppoz/svg/"):

        newLayer = qgis.core.QgsVectorLayer("Point?crs=EPSG:2180", layerName, "memory")
        #newLayer.dataProvider().setEncoding(u'System')
        svgFileName = "tower.svg"

        if color == "red":
            svgFileName = "tower_red.svg"
        elif color == "green":
            svgFileName = "tower_gre.svg"
        elif color == "blue":
            svgFileName = "tower_blu.svg"

        # symbole
        svgStyle = {}
        svgStyle['fill'] = '#ff0000'
        svgStyle['name'] = os.path.join(QtCore.QFileInfo(qgis.core.QgsApplication.qgisUserDatabaseFilePath()).path() + str(svgCatalog), svgFileName)
        svgStyle['outline'] = '#ff0000'
        svgStyle['outline-width'] = '6.8'
        svgStyle['size'] = '8'
        symLyr1 = qgis.core.QgsSvgMarkerSymbolLayer.create(svgStyle)
        newLayer.renderer().symbols(QgsRenderContext())[0].changeSymbolLayer(0, symLyr1)
        newLayer.dataProvider().addAttributes([qgis.core.QgsField("ID", QVariant.Int)])
        return newLayer

    @staticmethod
    def createSimpleLineMemLayer(layerName, color):
        newLayer = qgis.core.QgsVectorLayer("LineString?crs=EPSG:2180", layerName, "memory")
        #newLayer.dataProvider().setEncoding(u'System')
        # symbole
        symbol = qgis.core.QgsLineSymbol.createSimple({'width': '1', 'color': color, 'penstyle': 'dash'})
        newLayer.renderer().setSymbol(symbol)
        newLayer.dataProvider().addAttributes([qgis.core.QgsField("ID", QVariant.Int)])

        return newLayer

    @staticmethod
    def createSimplePolyMemLayer(layerName):
        newLayer = qgis.core.QgsVectorLayer("Polygon?crs=EPSG:2180", layerName, "memory")
        #newLayer.dataProvider().setEncoding(u'System')
        # symbole
        symbol = qgis.core.QgsFillSymbol.createSimple({'color_border': 'red', 'color': '255,253,0,127', 'style_border': 'solid', 'width_border': '1','style':'solid'})
        newLayer.renderer().setSymbol(symbol)
        newLayer.dataProvider().addAttributes([qgis.core.QgsField("ID", QVariant.Int)])

        return newLayer

    @staticmethod
    def createSimplePolyMemLayer2(layerName):
        newLayer = qgis.core.QgsVectorLayer("Polygon?crs=EPSG:2180", layerName, "memory")
        #newLayer.dataProvider().setEncoding(u'System')
        # symbole
        symbol = qgis.core.QgsFillSymbol.createSimple(
            {'color_border': 'red', 'color': '255,253,0,127', 'style_border': 'solid', 'width_border': '1',
             'style': 'solid'})

        newLayer.renderer().setSymbol(symbol)
        newLayer.dataProvider().addAttributes([qgis.core.QgsField("ID", QVariant.Int)])

        return newLayer

    @staticmethod
    def updateFeatureInLayer(layer, featureGeom, idval):
        
        with edit(layer):   
            for feat in layer.getFeatures():
                layer.deleteFeature(feat.id())
      
        # dodajemy feature
        if idval != -1:
            ft = qgis.core.QgsFeature()
            ft.setAttributes([1])
            ft.setGeometry(featureGeom)
            layer.dataProvider().addFeatures([ft])
        
        layer.commitChanges()
        layer.updateExtents()
        #print('updateFeatureInLayer')
        #except Exception as e:
        #print(e.Message)

    @staticmethod
    def isFloat(strval):
        ret = True
        try:
            val = float(strval)
        except ValueError:
            ret = False

        return ret

    @staticmethod
    def locatePointInPolyLayer(point, polyLayer):
        resultFeature = None

        try:
            allfeatures = {feature.id(): feature for (feature) in polyLayer.getFeatures()}

            for feature in list(allfeatures.values()):
                if point.intersects(feature.geometry()):
                    resultFeature = feature
                    break
        except Exception as e:
            print(e.Message)
        finally:
            return resultFeature

    @staticmethod
    def degMinSec(dd):
        if type(dd) != 'float':
            try:
                dd = float(dd)
            except:
                return 0

        is_positive = dd >= 0
        dd = abs(dd)
        minutes, seconds = divmod(dd * 3600, 60)
        degrees, minutes = divmod(minutes, 60)
        degrees = degrees if is_positive else -degrees
        return (int(degrees), int(minutes), round(seconds,2))

    @staticmethod
    def transformGeometry(crsSrc, crsDest, geom, prj):
        xform = qgis.core.QgsCoordinateTransform(crsSrc, crsDest,prj)
        return xform.transform(geom)

    @staticmethod
    def pointAlongLine(line_start, line_end, dist):
        # mid point = vector coordinates [x2-x1,y2-y1]
        pointm = qgis.core.QgsPointXY(line_end.x()-line_start.x(), line_end.y()-line_start.y())
        # direction cosines of the segment
        cosa, cosb = GeneralUtils.dircos(pointm)
        # length of the segment
        lg = GeneralUtils.length(line_end, line_start)
        return qgis.core.QgsPointXY(line_start.x()  + (dist * cosa), line_start.y() + (dist*cosb))

    @staticmethod
    def length(point1, point2):
        # with PyQGIS: sqrDist
        return math.sqrt(point1.sqrDist(point2))

    @staticmethod
    def mag(point):
        # magnitude of a vector
        return math.sqrt(point.x() ** 2 + point.y() ** 2)

    @staticmethod
    def dircos(point):
        cosa = point.x() / GeneralUtils.mag(point)
        cosb = point.y() / GeneralUtils.mag(point)
        return cosa, cosb
    
layerExists = GeneralUtils.layerExists
getFieldIndexFromLayer = GeneralUtils.getFieldIndexFromLayer
getUniqueForestryPrefixes = GeneralUtils.getUniqueForestryPrefixes
createPrjFile = GeneralUtils.createPrjFile
getPpozFeatures = GeneralUtils.getPpozFeatures
getParkiFeatures = GeneralUtils.getParkiFeatures
getParkiOddzWydzFeatures = GeneralUtils.getParkiOddzWydzFeatures
createPpozMemLayer = GeneralUtils.createPpozMemLayer
createPnMemLayer = GeneralUtils.createPnMemLayer
createSimplePointMemLayer = GeneralUtils.createSimplePointMemLayer
createSimpleLineMemLayer = GeneralUtils.createSimpleLineMemLayer
createSimplePolyMemLayer = GeneralUtils.createSimplePolyMemLayer
createPpozMemFireLayer = GeneralUtils.createPpozMemFireLayer

copyQmlStyle = GeneralUtils.copyQmlStyle
updateFeatureInLayer = GeneralUtils.updateFeatureInLayer
isFloat = GeneralUtils.isFloat
locatePointInPolyLayer = GeneralUtils.locatePointInPolyLayer
degMinSec = GeneralUtils.degMinSec
transformGeometry = GeneralUtils.transformGeometry
pointAlongLine = GeneralUtils.pointAlongLine